/*
 * Decompiled with CFR 0.152.
 */
package com.example.playerdatasync;

import com.example.playerdatasync.PlayerDataSync;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicInteger;

public class ConnectionPool {
    private final PlayerDataSync plugin;
    private final ConcurrentLinkedQueue<Connection> availableConnections;
    private final AtomicInteger connectionCount;
    private final int maxConnections;
    private final String databaseUrl;
    private final String username;
    private final String password;
    private volatile boolean shutdown = false;

    public ConnectionPool(PlayerDataSync plugin, String databaseUrl, String username, String password, int maxConnections) {
        this.plugin = plugin;
        this.databaseUrl = databaseUrl;
        this.username = username;
        this.password = password;
        this.maxConnections = maxConnections;
        this.availableConnections = new ConcurrentLinkedQueue();
        this.connectionCount = new AtomicInteger(0);
    }

    public Connection getConnection() throws SQLException {
        if (this.shutdown) {
            throw new SQLException("Connection pool is shut down");
        }
        Connection connection = this.availableConnections.poll();
        if (connection != null && this.isConnectionValid(connection)) {
            return connection;
        }
        if (this.connectionCount.get() < this.maxConnections && (connection = this.createNewConnection()) != null) {
            this.connectionCount.incrementAndGet();
            this.plugin.getLogger().fine("Created new database connection. Pool size: " + this.connectionCount.get());
            return connection;
        }
        long startTime = System.currentTimeMillis();
        long waitTime = 10L;
        long maxWaitTime = 100L;
        long totalTimeout = 10000L;
        while (System.currentTimeMillis() - startTime < 10000L) {
            connection = this.availableConnections.poll();
            if (connection != null && this.isConnectionValid(connection)) {
                return connection;
            }
            try {
                Thread.sleep(waitTime);
                waitTime = Math.min(waitTime * 2L, 100L);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new SQLException("Interrupted while waiting for connection");
            }
        }
        this.plugin.getLogger().severe("Connection pool exhausted. " + this.getStats());
        throw new SQLException("Unable to obtain database connection within timeout (10000ms)");
    }

    public void returnConnection(Connection connection) {
        if (connection == null || this.shutdown) {
            try {
                if (connection != null) {
                    connection.close();
                    this.connectionCount.decrementAndGet();
                }
            }
            catch (SQLException e) {
                this.plugin.getLogger().warning("Error closing connection: " + e.getMessage());
            }
            return;
        }
        if (this.isConnectionValid(connection)) {
            this.availableConnections.offer(connection);
        } else {
            try {
                connection.close();
                this.connectionCount.decrementAndGet();
            }
            catch (SQLException e) {
                this.plugin.getLogger().warning("Error closing invalid connection: " + e.getMessage());
            }
        }
    }

    private boolean isConnectionValid(Connection connection) {
        try {
            return connection != null && !connection.isClosed() && connection.isValid(2);
        }
        catch (SQLException e) {
            return false;
        }
    }

    private Connection createNewConnection() {
        try {
            if (this.username != null && this.password != null) {
                return DriverManager.getConnection(this.databaseUrl, this.username, this.password);
            }
            return DriverManager.getConnection(this.databaseUrl);
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to create new database connection: " + e.getMessage());
            return null;
        }
    }

    public void initialize() {
        int initialConnections = Math.min(3, this.maxConnections);
        for (int i = 0; i < initialConnections; ++i) {
            Connection connection = this.createNewConnection();
            if (connection == null) continue;
            this.availableConnections.offer(connection);
            this.connectionCount.incrementAndGet();
        }
        this.plugin.getLogger().info("Connection pool initialized with " + this.availableConnections.size() + " connections");
    }

    public void shutdown() {
        Connection connection;
        this.shutdown = true;
        while ((connection = this.availableConnections.poll()) != null) {
            try {
                connection.close();
                this.connectionCount.decrementAndGet();
            }
            catch (SQLException e) {
                this.plugin.getLogger().warning("Error closing connection during shutdown: " + e.getMessage());
            }
        }
        this.plugin.getLogger().info("Connection pool shut down. Remaining connections: " + this.connectionCount.get());
    }

    public String getStats() {
        return String.format("Pool stats: %d/%d connections, %d available", this.connectionCount.get(), this.maxConnections, this.availableConnections.size());
    }
}

