/*
 * Decompiled with CFR 0.152.
 */
package com.example.playerdatasync;

import com.example.playerdatasync.PlayerDataSync;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import org.bukkit.Bukkit;
import org.bukkit.plugin.Plugin;
import org.bukkit.scheduler.BukkitTask;

public class BackupManager {
    private final PlayerDataSync plugin;
    private BukkitTask backupTask;
    private final SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd_HH-mm-ss");

    public BackupManager(PlayerDataSync plugin) {
        this.plugin = plugin;
    }

    public void startAutomaticBackups() {
        if (!this.plugin.getConfigManager().isBackupEnabled()) {
            return;
        }
        int intervalMinutes = this.plugin.getConfigManager().getBackupInterval();
        if (intervalMinutes <= 0) {
            return;
        }
        long ticks = (long)intervalMinutes * 60L * 20L;
        this.backupTask = Bukkit.getScheduler().runTaskTimerAsynchronously((Plugin)this.plugin, () -> {
            try {
                this.createBackup("automatic");
            }
            catch (Exception e) {
                this.plugin.getLogger().severe("Automatic backup failed: " + e.getMessage());
            }
        }, ticks, ticks);
        this.plugin.getLogger().info("Automatic backups enabled with " + intervalMinutes + " minute intervals");
    }

    public void stopAutomaticBackups() {
        if (this.backupTask != null) {
            this.backupTask.cancel();
            this.backupTask = null;
            this.plugin.getLogger().info("Automatic backups disabled");
        }
    }

    public CompletableFuture<BackupResult> createBackup(String type) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                String timestamp = this.dateFormat.format(new Date());
                String backupName = "backup_" + type + "_" + timestamp;
                File backupDir = new File(this.plugin.getDataFolder(), "backups");
                if (!backupDir.exists()) {
                    backupDir.mkdirs();
                }
                File backupFile = new File(backupDir, backupName + ".zip");
                try (ZipOutputStream zipOut = new ZipOutputStream(new FileOutputStream(backupFile));){
                    this.backupDatabase(zipOut, backupName);
                    this.backupConfiguration(zipOut, backupName);
                    this.backupLogs(zipOut, backupName);
                }
                this.cleanOldBackups();
                this.plugin.getLogger().info("Backup created: " + backupFile.getName());
                return new BackupResult(true, backupFile.getName(), backupFile.length());
            }
            catch (Exception e) {
                this.plugin.getLogger().severe("Backup creation failed: " + e.getMessage());
                return new BackupResult(false, null, 0L);
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void backupDatabase(ZipOutputStream zipOut, String backupName) throws SQLException, IOException {
        Connection connection = this.plugin.getConnection();
        if (connection == null) {
            throw new SQLException("No database connection available");
        }
        try {
            StringBuilder sqlDump;
            String tableName;
            block35: {
                tableName = this.plugin.getTablePrefix();
                sqlDump = new StringBuilder();
                sqlDump.append("-- PlayerDataSync Database Backup\n");
                sqlDump.append("-- Created: ").append(new Date()).append("\n\n");
                DatabaseMetaData metaData = connection.getMetaData();
                try (ResultSet rs = metaData.getTables(null, null, tableName, new String[]{"TABLE"});){
                    if (!rs.next()) break block35;
                    sqlDump.append("CREATE TABLE IF NOT EXISTS ").append(tableName).append(" (\n");
                    try (ResultSet columns = metaData.getColumns(null, null, tableName, null);){
                        ArrayList<String> columnDefs = new ArrayList<String>();
                        while (columns.next()) {
                            String columnName = columns.getString("COLUMN_NAME");
                            String dataType = columns.getString("TYPE_NAME");
                            int columnSize = columns.getInt("COLUMN_SIZE");
                            String nullable = columns.getString("IS_NULLABLE");
                            StringBuilder columnDef = new StringBuilder("  ").append(columnName).append(" ");
                            if (dataType.equals("VARCHAR")) {
                                columnDef.append("VARCHAR(").append(columnSize).append(")");
                            } else {
                                columnDef.append(dataType);
                            }
                            if ("NO".equals(nullable)) {
                                columnDef.append(" NOT NULL");
                            }
                            columnDefs.add(columnDef.toString());
                        }
                        sqlDump.append(String.join((CharSequence)",\n", columnDefs));
                    }
                    sqlDump.append("\n);\n\n");
                }
            }
            try (PreparedStatement ps = connection.prepareStatement("SELECT * FROM " + tableName);
                 ResultSet rs = ps.executeQuery();){
                while (rs.next()) {
                    sqlDump.append("INSERT INTO ").append(tableName).append(" VALUES (");
                    ResultSetMetaData rsMeta = rs.getMetaData();
                    ArrayList<Object> values = new ArrayList<Object>();
                    for (int i = 1; i <= rsMeta.getColumnCount(); ++i) {
                        String value = rs.getString(i);
                        if (value == null) {
                            values.add("NULL");
                            continue;
                        }
                        values.add("'" + value.replace("'", "''") + "'");
                    }
                    sqlDump.append(String.join((CharSequence)", ", values));
                    sqlDump.append(");\n");
                }
            }
            zipOut.putNextEntry(new ZipEntry(backupName + "/database.sql"));
            zipOut.write(sqlDump.toString().getBytes());
            zipOut.closeEntry();
        }
        finally {
            this.plugin.returnConnection(connection);
        }
    }

    private void backupConfiguration(ZipOutputStream zipOut, String backupName) throws IOException {
        File[] messageFiles;
        File configFile = new File(this.plugin.getDataFolder(), "config.yml");
        if (configFile.exists()) {
            zipOut.putNextEntry(new ZipEntry(backupName + "/config.yml"));
            Files.copy(configFile.toPath(), zipOut);
            zipOut.closeEntry();
        }
        if ((messageFiles = this.plugin.getDataFolder().listFiles((dir, name) -> name.startsWith("messages_") && name.endsWith(".yml"))) != null) {
            for (File messageFile : messageFiles) {
                zipOut.putNextEntry(new ZipEntry(backupName + "/" + messageFile.getName()));
                Files.copy(messageFile.toPath(), zipOut);
                zipOut.closeEntry();
            }
        }
    }

    private void backupLogs(ZipOutputStream zipOut, String backupName) throws IOException {
        File logsDir = new File(this.plugin.getDataFolder(), "logs");
        if (logsDir.exists()) {
            Files.walk(logsDir.toPath(), new FileVisitOption[0]).filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).forEach(logFile -> {
                try {
                    String relativePath = logsDir.toPath().relativize((Path)logFile).toString();
                    zipOut.putNextEntry(new ZipEntry(backupName + "/logs/" + relativePath));
                    Files.copy(logFile, zipOut);
                    zipOut.closeEntry();
                }
                catch (IOException e) {
                    this.plugin.getLogger().warning("Failed to backup log file: " + String.valueOf(logFile) + " - " + e.getMessage());
                }
            });
        }
    }

    private void cleanOldBackups() {
        int keepBackups = this.plugin.getConfigManager().getBackupsToKeep();
        File backupDir = new File(this.plugin.getDataFolder(), "backups");
        if (!backupDir.exists()) {
            return;
        }
        File[] backupFiles = backupDir.listFiles((dir, name) -> name.endsWith(".zip"));
        if (backupFiles == null || backupFiles.length <= keepBackups) {
            return;
        }
        Arrays.sort(backupFiles, Comparator.comparingLong(File::lastModified));
        int toDelete = backupFiles.length - keepBackups;
        for (int i = 0; i < toDelete; ++i) {
            if (!backupFiles[i].delete()) continue;
            this.plugin.getLogger().info("Deleted old backup: " + backupFiles[i].getName());
        }
    }

    public List<BackupInfo> listBackups() {
        ArrayList<BackupInfo> backups = new ArrayList<BackupInfo>();
        File backupDir = new File(this.plugin.getDataFolder(), "backups");
        if (!backupDir.exists()) {
            return backups;
        }
        File[] backupFiles = backupDir.listFiles((dir, name) -> name.endsWith(".zip"));
        if (backupFiles == null) {
            return backups;
        }
        for (File backupFile : backupFiles) {
            backups.add(new BackupInfo(backupFile.getName(), backupFile.length(), new Date(backupFile.lastModified())));
        }
        backups.sort(Comparator.comparing(BackupInfo::getCreatedDate).reversed());
        return backups;
    }

    public CompletableFuture<Boolean> restoreFromBackup(String backupName) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                File backupFile = new File(this.plugin.getDataFolder(), "backups/" + backupName);
                if (!backupFile.exists()) {
                    this.plugin.getLogger().severe("Backup file not found: " + backupName);
                    return false;
                }
                this.plugin.getLogger().info("Restore from backup: " + backupName + " (not implemented yet)");
                return true;
            }
            catch (Exception e) {
                this.plugin.getLogger().severe("Restore failed: " + e.getMessage());
                return false;
            }
        });
    }

    public static class BackupInfo {
        private final String fileName;
        private final long fileSize;
        private final Date createdDate;

        public BackupInfo(String fileName, long fileSize, Date createdDate) {
            this.fileName = fileName;
            this.fileSize = fileSize;
            this.createdDate = createdDate;
        }

        public String getFileName() {
            return this.fileName;
        }

        public long getFileSize() {
            return this.fileSize;
        }

        public Date getCreatedDate() {
            return this.createdDate;
        }

        public String getFormattedSize() {
            if (this.fileSize < 1024L) {
                return this.fileSize + " B";
            }
            if (this.fileSize < 0x100000L) {
                return String.format("%.1f KB", (double)this.fileSize / 1024.0);
            }
            return String.format("%.1f MB", (double)this.fileSize / 1048576.0);
        }
    }

    public static class BackupResult {
        private final boolean success;
        private final String fileName;
        private final long fileSize;

        public BackupResult(boolean success, String fileName, long fileSize) {
            this.success = success;
            this.fileName = fileName;
            this.fileSize = fileSize;
        }

        public boolean isSuccess() {
            return this.success;
        }

        public String getFileName() {
            return this.fileName;
        }

        public long getFileSize() {
            return this.fileSize;
        }
    }
}

